#!/bin/bash
set -e

VENV_DIR="venv"

echo "==============================================================="
echo "   CLIP Concept Viewer - Setup Script - by https://m14w.com/"
echo "==============================================================="
echo ""

# Detect CUDA version
CUDA_VERSION="none"
CUDA_MAJOR=0
CUDA_MINOR=0

if command -v nvcc &> /dev/null; then
    CUDA_VERSION=$(nvcc --version | grep "release" | sed -n 's/.*release \([0-9]*\.[0-9]*\).*/\1/p')
fi

if [ "$CUDA_VERSION" = "none" ] || [ -z "$CUDA_VERSION" ]; then
    if command -v nvidia-smi &> /dev/null; then
        CUDA_VERSION=$(nvidia-smi | grep "CUDA Version" | sed -n 's/.*CUDA Version: \([0-9]*\.[0-9]*\).*/\1/p')
    fi
fi

if [ -z "$CUDA_VERSION" ]; then
    CUDA_VERSION="none"
fi

echo "Detected CUDA version: $CUDA_VERSION"
echo ""

# Determine recommended PyTorch CUDA version
PYTORCH_CUDA="cu121"
PYTORCH_CUDA_DESC="CUDA 12.1"

if [ "$CUDA_VERSION" != "none" ]; then
    CUDA_MAJOR=$(echo "$CUDA_VERSION" | cut -d. -f1)
    CUDA_MINOR=$(echo "$CUDA_VERSION" | cut -d. -f2)
    
    if [ "$CUDA_MAJOR" -ge 12 ] && [ "$CUDA_MINOR" -ge 8 ]; then
        PYTORCH_CUDA="cu128"
        PYTORCH_CUDA_DESC="CUDA 12.8"
    elif [ "$CUDA_MAJOR" -ge 12 ] && [ "$CUDA_MINOR" -ge 4 ]; then
        PYTORCH_CUDA="cu124"
        PYTORCH_CUDA_DESC="CUDA 12.4"
    elif [ "$CUDA_MAJOR" -ge 12 ]; then
        PYTORCH_CUDA="cu121"
        PYTORCH_CUDA_DESC="CUDA 12.1"
    elif [ "$CUDA_MAJOR" -eq 11 ]; then
        PYTORCH_CUDA="cu118"
        PYTORCH_CUDA_DESC="CUDA 11.8"
    fi
fi

echo "Available PyTorch builds:"
echo "  1. $PYTORCH_CUDA_DESC [Recommended based on detection]"
echo "  2. CUDA 12.8"
echo "  3. CUDA 12.4"
echo "  4. CUDA 12.1"
echo "  5. CUDA 11.8"
echo "  6. CPU only (no GPU acceleration)"
echo ""

read -p "Select PyTorch version [1-6] (default=1): " CHOICE
CHOICE=${CHOICE:-1}

case $CHOICE in
    2) PYTORCH_CUDA="cu128"; PYTORCH_CUDA_DESC="CUDA 12.8" ;;
    3) PYTORCH_CUDA="cu124"; PYTORCH_CUDA_DESC="CUDA 12.4" ;;
    4) PYTORCH_CUDA="cu121"; PYTORCH_CUDA_DESC="CUDA 12.1" ;;
    5) PYTORCH_CUDA="cu118"; PYTORCH_CUDA_DESC="CUDA 11.8" ;;
    6) PYTORCH_CUDA="cpu"; PYTORCH_CUDA_DESC="CPU only" ;;
esac

echo ""
echo "Selected: $PYTORCH_CUDA_DESC"
echo ""

if [ -d "$VENV_DIR" ] && [ -f "$VENV_DIR/bin/activate" ]; then
    echo "Virtual environment already exists."
    read -p "Recreate it? (y/N): " RECREATE
    if [ "$RECREATE" = "y" ] || [ "$RECREATE" = "Y" ]; then
        echo "Removing old venv..."
        rm -rf "$VENV_DIR"
    else
        source "$VENV_DIR/bin/activate"
        echo "Venv activated. Skipping to dependency check..."
        goto_deps=true
    fi
fi

if [ -z "$goto_deps" ]; then
    echo "Creating virtual environment..."
    python3 -m venv "$VENV_DIR"
    
    echo "Activating virtual environment..."
    source "$VENV_DIR/bin/activate"
fi

echo "Upgrading pip..."
pip install --upgrade pip

echo ""
echo "Installing PyTorch ($PYTORCH_CUDA_DESC)..."
if [ "$PYTORCH_CUDA" = "cpu" ]; then
    pip install torch torchvision --index-url https://download.pytorch.org/whl/cpu
else
    pip install torch torchvision --index-url https://download.pytorch.org/whl/$PYTORCH_CUDA
fi

echo ""
echo "Installing other dependencies..."
pip install open-clip-torch transformers pillow psutil sentencepiece protobuf

echo ""
echo "============================================"
echo "   Setup complete!"
echo "============================================"
echo ""
echo "To run the script:"
echo "  1. Run: source activate.sh"
echo "  2. Run: python clip_picture_search_engine.py [options]"
echo ""
echo "Options:"
echo "  --b32      Use lighter ViT-B/32 model"
echo "  --sigclip  Use SigLIP model"
echo "  --fast     Enable float16 + torch.compile"
echo "  --sub      Scan subdirectories"
echo ""
